#!/usr/bin/perl
#
# Storage module for Acctfind.pm based on DBD::SQLite
#
use strict;
use warnings;


package Port25::Acctfind::Storage::SQLite;

use vars qw(@ISA);

@ISA = ("Port25::Acctfind::Storage");


# not using "use" to prevent rpm from automatically creating a dependecy :-(
BEGIN { require DBI; DBI->import();
        require DBD::SQLite; DBD::SQLite->import();
        # prevent warning '"DBD::SQLite::sqlite_version" used only once'
        my $dummy = $DBD::SQLite::sqlite_version;
};


sub new {
    my $class = shift @_;
    my $tmpDir = shift @_;
    
    my $self = $class->SUPER::new($tmpDir);
    
    $self->{acctfind} = shift @_;
    
    my $dbh = DBI->connect("dbi:SQLite:dbname=$self->{dbFile}","","")
        or die $DBI::errstr;;
        
    $self->{dbh} = $dbh;
    
    $dbh->do("create table acctfind (key blob primary key asc, value blob)")
        or die $dbh->errstr;
        
    my $getSth = $dbh->prepare("select value, key from acctfind order by key asc")
        or die $dbh->errstr;
    my $putSth = $dbh->prepare("insert into acctfind (key, value) values (?, ?)")
        or die $dbh->errstr;
        
    $self->{getSth} = $getSth;
    $self->{putSth} = $putSth;
    
    $self->{dbEntries} = 0;
    $self->{dbEntryCounter} = "a" x 8, # About 8 billion
    
    return $self;
}


sub DESTROY {
    my $self = shift @_;
    
    $self->{getSth}->finish() if $self->{getSth};
    $self->{putSth}->finish() if $self->{putSth};
    delete $self->{getSth};
    delete $self->{putSth};
    $self->{dbh}->disconnect() if $self->{dbh};
    delete $self->{dbh};
    
    undef $self->{db};
    untie $self->{records};
#    $self->unlinkDbFile();
    
    $self->SUPER::DESTROY(@_);
}


#
# Loads a record from the data store
#
# @param  $key  the key of the last item returned, or '' if first call
# @param  $n    0 for the first call, !0 for succeeding calls
# @return  $status  0 for success, !0 for errors
# @return  $key     the key of the item returned
# @return  $rec     the record loaded
#
sub loadRecord {
    my $self = shift @_;
    my $key = shift @_;
    my $n = shift @_;
    
    if ($n == 0) {
        if (! defined $self->{getSth}->execute()) {
            warn $self->{getSth}->errstr();
            return $self->{getSth}->err(), $key, undef; # end of db
        }
    }
    
    my $row;
    unless ($row = $self->{getSth}->fetchrow_arrayref()) {
        warn $self->{getSth}->errstr() if $self->{getSth}->err();
        return 1, $key, undef; # end of db
    }
    
    my $chunk = $row->[0];
    $key = $row->[1];
    
    my $rec = $self->{acctfind}->thawRec($chunk) if $chunk;
    
    return 0, $key, $rec;
}


sub storeRecord {
    my $self = shift @_;
    my $rec = shift @_;
    
    $self->{dbEntryCounter}++;
    my $key = lc $self->generateSortingKeyForRec($rec);
    
    my $chunk = $self->{acctfind}->freezeRec($rec);
    
    # add frozen record to SQLite
    # downcase keys to make sorting case independent
    $self->{putSth}->execute(lc $key, $chunk);
    
    $self->{dbEntries}++;
    
    return $self->{putSth}->err() ? $self->{putSth}->errstr() : undef;
}


1;

