#!/usr/bin/perl -w
#
# Storage module for Acctfind.pm based on DB_File
#
use strict;

package Port25::Acctfind::Storage::DBFile;

use vars qw(@ISA);

@ISA = ("Port25::Acctfind::Storage");


# not using "use" to prevent rpm from automatically creating a dependency :-(
BEGIN { require DB_File; DB_File->import(); };


sub new {
    my $class = shift @_;
    my $tmpDir = shift @_;
    
    my $self = $class->SUPER::new($tmpDir);
    
    $self->{acctfind} = shift @_;
    
    my %records;
    $self->{db} = tie %records, "DB_File", "$self->{dbFile}",
                                O_RDWR|O_CREAT, 0666, $DB_BTREE
        or die "$0: could not open DB file $self->{dbFile}: $!\n";
        
    $self->{records} = %records;
    
    $self->{dbEntries} = 0;
    $self->{dbEntryCounter} = "a" x 8, # About 8 billion
    
    return $self;
}


sub DESTROY {
    my $self = shift @_;
    
    undef $self->{db};
    untie $self->{records};
#    $self->unlinkDbFile();
    
    $self->SUPER::DESTROY(@_);
}


#
# Loads a record from the data store
#
# @param  $key  the key of the last item returned, or '' if first call
# @param  $n    0 for the first call, !0 for succeeding calls
# @return  $status  0 for success, !0 for errors
# @return  $key     the key of the item returned
# @return  $rec     the record loaded
#
sub loadRecord {
    my $self = shift @_;
    my $key = shift @_;
    my $n = shift @_;
    
    my $r_cursor = ($n == 0) ? R_FIRST : R_NEXT;
    
    # retrieve frozen record from DB_File
    my $chunk = '';
    my $status = $self->{db}->seq($key, $chunk, $r_cursor);
    
    my $rec = $self->{acctfind}->thawRec($chunk) if $chunk and $status == 0;
    
    return $status, $key, $rec;
}


sub storeRecord {
    my $self = shift @_;
    my $rec = shift @_;
    
    my $db = $self->{db};
    my $status;
    my $key;
    
    $self->{dbEntryCounter}++;
    $key = $self->generateSortingKeyForRec($rec);
    
    my $chunk = $self->{acctfind}->freezeRec($rec);
    
    # add frozen record to DB_File
    # downcase keys to make sorting case independent
    $status = $db->put(lc $key, $chunk);
    
    $self->{dbEntries}++;
    
    return $status != 0 ? $! : undef;
}


1;

