#!/usr/bin/perl -w
#
# Super class of the storage modules for Acctfind.pm
#
use strict;

package Port25::Acctfind::Storage;

use File::Temp qw/:mktemp/;
use Cwd;


sub new {
    my ($class, $tmpDir, $acctfind) = @_;
    
    $tmpDir = "." unless $tmpDir;
    $tmpDir = "/tmp" if $tmpDir eq "/";
    
    my $self = bless {}, $class;
    
    my $dirSep = ($^O eq 'MSWin32') ? "\\" : "/";
    my $quotedDirSep = $dirSep;
    $quotedDirSep =~ s/\\/\\\\/g;
    
    $tmpDir .= $dirSep unless $tmpDir =~ /$quotedDirSep$/;
    
    $self->{tmpDir} = $tmpDir;
    $self->{acctfind} = $acctfind;
    
    $self->createDbFileName();
    
    return $self;
}


sub DESTROY {
    my $self = shift@_;
    
    unlink $self->{dbFile} if $self->{dbFile};
    delete $self->{dbFile};
}


sub dbFile {
    my $self = shift @_;
    
    return $self->{dbFile};
}


#
# Creates a name for a file that does not exist in the temporary directory
# and stores it in $self->{dbFile}. Does not do anything if this setting
# already exists
#
sub createDbFileName {
    my $self = shift @_;
     
    return if $self->{dbFile};
    
    my $dbFile;
    
    do {
        $dbFile = mktemp("$self->{tmpDir}acctfind-XXXXX");
    }
    while (-f $dbFile);
    
    $self->{dbFile} = $dbFile;
}



#
# Configures the information about sorting
#
# @param  $sortingTag    the tag to sort by
# @param  $sortByDomain  a flag indicating whether to sort by domain,
#                        i.e. ignore anything before a '@'
#
sub setupSorting {
    my $self = shift @_;
    my ($sortingTag, $sortByDomain) = @_;
    
    if ($sortingTag =~ /^record.(arf|header)\s*:\s*(.*)/i) {
        $sortingTag = "internal:$1";
        $self->{sortingField} = $2;
    }
    
    $self->{sortingTag} = $sortingTag;
    $self->{sortByDomain} = $sortByDomain;
}


#
# generates a (unique) key for sorting the records
#
# @param   $rec      a reference to the record to generate the key for
# @return  the generated key
#
sub generateSortingKeyForRec {
    my ($self, $rec) = @_;
    
    my $key;
    
    if (exists $self->{sortingTag}) {
        if ($self->{sortingTag} =~ /^internal:(arf|header)$/) {
            my $fields = $rec->{'record.' . $self->{acctfind}->getTag($1)};
            foreach my $field (@$fields) {
                if ($field->{name} eq $self->{sortingField}) {
                    $key = $field->{content};
                    last;
                }
            }
        }
        else {
            $key = $rec->{$self->{sortingTag}};
        }
    }
    
    if ($key) {
        $key =~ s/.*@/@/ if $self->{sortByDomain};
    }
    else {
        $key = '';
    }
    
    # make the keys unique and the sorting stable
    $key .= "\0$self->{dbEntryCounter}";
    
    return $key;
}


1;

